{*********************************************************************************************
 **                          Delphi Asteroids Vector Graphics Demo                          **
 *********************************************************************************************
 **   			      							Programmed By:	John Ayres                                  **
 **			      									Date:						July 23, 1996                               **
 **			      									Splash Screen:	David Bowden                                **
 ** 			      							Presented to the Delphi Developers                            **
 **                           of Dallas User's Group on July 31,                            **
 **                           1996, in a demonstration of graphics                          **
 **                           programming in Delphi, with a                                 **
 **                           concentration on games development.                           **
 *********************************************************************************************
 **  I wanted a really good example of how a complete game could be done with Delphi.  My   **
 **  presentation was centered around sprite movement techniques, and I think this program  **
 **  serves as a good example of how such techniques can be combined with object oriented   **
 **  programming to create a full fledged game.  Granted, this is not a commercial quality  **
 **  game (if it were, I wouldn't be giving it away), but with a little optimization in the **
 **  correct places, its speed could be improved drastically, and it would make a good      **
 **  starting point for beginning game programmers.  Hopefully, this will at the very least **
 **  serve to show that Delphi can do more than just databases.                             **
 **                                                                                         **
 **	 This code was originally written in 16 bit.  It will run under Delphi 2, but with a    **
 **  few modifications.  For one, hook functions are implemented differently.  There is     **
 **  no need to do a MakeProcInstance, you can simply pass the address of the keyboard      **
 **  hook function.  Look at Steve Teixiera's 'Delphi 2 Developers Guide' for a good        **
 **  example of how to implement hooks under Delphi 2.  The IntersectRect functions under   **
 **  Delphi 2 return a boolean now, instead of an integer.  And for some reason, in the     **
 **  DrawPolygon method, you must increase the NumVerts by one when calling the PolyLine    **
 **  function, or the asteroids do not get closed.  This may be an oversight on my part,    **
 **  but as of this writing I am running out of time and I have a lot more work left to     **
 **  do for my presentation.  If anyone finds a better explaination of why this is          **
 **  necessary, I would love to hear from you.                                              **
 *********************************************************************************************
 **  Thanks to all of those programmers who generously provide free examples to      	  	  **
 **  those of us still learning.  In carrying on such a theme, this program is hereby       **
 **  donated to the public domain, in the hopes that I will contribute to someone's         **
 **  knowledge base, and see Delphi being used in a much broader arena.                     **
 *********************************************************************************************
 **                          --- Delphi Developers of Dallas ---                            **
 **                                                                                         **
 **  If you would like more information on the Delphi Developers of Dallas, or would like   **
 **  to see the other information that was distributed during my presentation, please       **
 **  visit our web site at:                                                                 **
 **	                                       www.3-D.org                                  		**
 **                                                                                         **
 **  Or, alternatively, feel free to write me at:                                           **
 **                                                                                         **
 **																			John Ayres                                          **
 **															Director of Vendor Relations                                **
 **                               Delphi Developers of Dallas                               **
 **                                102447.10@Compuserve.Com                                 **
 *********************************************************************************************
 **  																		Shameless Plug:                                     **
 **  John Ayres is the Executive Director of Software Development for Puzzled Software,     **
 **  Inc.  At Puzzled Software, we use Delphi exclusively to create add-ins and utilities   **
 **	 for major CADD packages.  If you would like to know more about Puzzled Software,       **
 **  please visit our web page at:                                                          **
 **                                                                                         **
 **            		  			www.numera.com/3rdparty/puzzled/puzzled.htm                       **
 *********************************************************************************************}
unit Asteru;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Menus, ExtCtrls, TestTrig, StdCtrls, Aboutu, MMSystem;

{global constants}
const
	MISSLEFRAMELIFE = 30;			{number of frames the missle will stay alive}
  NUMMISSLES = 20;					{total number of missles allowed in the game}
  MISSLEVELDELTA = 10;			{a constant to determine the missles velocity in relation to the
  													 firing ship}
  NUMPARTICLES = 29;				{Number of particles in a particle system}
  EXHAUSTLIFESPAN = 10;			{Number of frames that ship exhaust is on the screen}
  EXHAUSTVELDELTA = 2;			{A constant for adjusting the exhaust velocity in relation to the
  													 player ship velocity}
  NUMASTEROIDS = 21;				{the total number of asteroids allowed}
  MAXASTEROIDRAD = 40;			{the maximum asteroid radius}
  NUMEXLPDS = 9;						{maximum number of explosions on the screen at once}
  EXPLDLIFE = 10;						{lifespan of explosion particles}
  EXPLDVELDELTA = 4;				{velocitiy adjustment for explosion particles}
  SHIELDLIFESPAN = 30;			{the number of frames that the players shields hold out}
  LEVELPAUSE = 40;					{the number of frames to pause between levels}

{This array holds colors...}
type
	TFadeColors = array[1..5] of TColor;

{for the fade effect used on bullets and exhaust particles}
const
  FadeColors: TFadeColors = (clMaroon, clRed, clOlive, clYellow, clWhite);

type
  {this is a real number based point, for use in polygon vertex tracking}
  TRPoint = record
  	X, Y: Real;
  end;

  TAsteroidForm = class(TForm)
    MainMenu1: TMainMenu;
    Panel1: TPanel;
    Panel2: TPanel;
    Panel3: TPanel;
    Panel4: TPanel;
    PlayField: TPaintBox;
    File1: TMenuItem;
    NewGame1: TMenuItem;
    N1: TMenuItem;
    Exit1: TMenuItem;
    Help1: TMenuItem;
    About1: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure NewGame1Click(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
  private
    { Private declarations }
    FDoLoop: Boolean;                     {Controls the processing of the main loop}
    FOffscreenBuffer: TBitmap;						{Double Buffer for flicker-free animation}
  public
    { Public declarations }
    procedure MainLoop;										{Our game control loop function}

    {this procedure will start a new bullet for both player and enemy}
    procedure StartMissle(InitialX, InitialY: Real; Facing: Integer; IsPlayer: Boolean;
    											StartXVel, StartYVel: Real);

    {move all the missles, do collision check, and draw them}
    procedure MoveMissles;

    {This generic procedure takes any polygonal shape and draws it at the specified rotation}
    procedure DrawPolygon(const ThePolygon: Array of TRPoint; NumVerts: integer;
    											TheAngle: Integer; XPos, YPos: Real; TheColor: TColor);

    {this will do all movement for the players ship}
    procedure MovePlayerShip;

    {Start some of the particles for the ship exhaust effect}
    procedure StartShipExhaustBurst;

    {draws and moves the ship exhaust effect}
    procedure DrawShipExhaust;

    {this procedure will start an asteroid with a relative diameter}
    procedure StartAsteroid(InitialX,InitialY: Integer; Facing: Integer; Radius: Integer);

    {this procedure moves and draws all the asteroids}
    procedure MoveAsteroids;

    {Start an explosion particle system}
    procedure StartExplosion(InitialX,InitialY: Integer);

    {Draw the explosion particle systems}
    procedure DrawExplosions;

    {this procedure starts a new player out in the middle, with shields on}
    procedure StartPlayer;

    {this procedure simply throws us into the main loop when the program is waiting}
    procedure Idling(Sender: TObject; var Done: Boolean);

    {this is used to clear out all of the arrays so we can start new games}
    procedure ClearAll;

    {this procedure will essentially end the game and put us in Intermission mode}
    procedure EndGame;
  end;

  {the base moving graphic object all others will be based on. all sprites in this game
   will be vector based}
  TSprite = class
  	XPos, YPos: Real; 										{world coordinates, center of sprite}
    XVel, YVel: Real;                     {world velocities}
    Living: Boolean;                      {if it's alive, it'll be moved}
    Color: TColor;												{the color the sprite will be drawn in}
  end;

  {This is used for missles and particle systems}
  TParticle = class(TSprite)
  	Lifespan: Integer;										{expressed in terms of number of frames}
    ColDelta: Integer;										{bounding box for missle types}
  end;

  {this will be for all collision enabled sprites, those with bounding boxes, and those
   with an actual polygon associated with them}
  TCollidable = class(TSprite)
    ColDelta: Integer;                    {used to determine a bounding box from the center}
    ThePolygon: array[0..19] of TRPoint;  {the polygon points for drawing the shape}
    NumVertices: integer;									{the number of vertices used in ThePolygon}
    Angle: Integer;												{rotation the sprite is facing}
  end;

  {This is the class for the players ship.  It simply adds an indicator for the shields}
  TPlayerShip = class(TCollidable)
  	ShieldsOn: Boolean;										{will be true when the shields are up}
    ShieldLife: Integer;									{holds the current count of the shields lifespan}
  end;

  {Asteroids have a few more properties other objects don't}
  TAsteroid = class(TCollidable)
  	RotationRate: Integer;								{the angle it rotates through each frame}
  end;

  TParticleSystem = array[0..NUMPARTICLES] of TParticle;  {used for explosions and ship exhaust}

  TExplosion = record
  	Living: boolean;
    Particles: TParticleSystem;
  end;

var
  AsteroidForm: TAsteroidForm;
  PlayerShip: TPlayerShip;
  Asteroids: array[0..NUMASTEROIDS] of TAsteroid; {the array of asteroids}
  Missles: array[0..NUMMISSLES] of TParticle;			{the last two missles are enemy's only}
  Explosions: array[0..NUMEXLPDS] of TExplosion;	{player and asteroid}
  ShipExhaust: TParticleSystem;                 	{exhause effect from the players ship (fluff)}
  KBHook: HHook;																	{this intercepts keyboard input}
  Score: Longint;																	{The players score}
  NumShipsLeft: Integer;													{The number of lives the player has left}
  CurLevel: Integer;															{the current level in the game}
  AnyAsteroidsMoved: Boolean;											{this will determine if all asteroids are dead}

  function KeyboardHook(Code: Integer; WordParam: Word; LongParam: LongInt): LongInt; export;

implementation

{$R *.DFM}

{We want to do this inside of a regular loop as opposed to putting this on a timer
 event.  This is becuase even with an interval of 1, the timer is too slow, and a
 loop will give us the best performance.}
procedure TAsteroidForm.MainLoop;
var
	LevelPauseCounter: Integer;		{for timing how long the level intermission has lasted}
  Count: Integer;								{general loop control}
begin
	while FDoLoop do
  begin
    {if the player still has ships, keep going with main loop}
    if NumShipsLeft>-1 then
    begin
      if AnyAsteroidsMoved then
      begin
	      {Erase the former frame, so we can begin a new one}
		    With FOffscreenBuffer.Canvas do
		    begin
		    	Brush.Color:=clBlack;
		    	Brush.Style:=bsSolid;
		    	Fillrect(Cliprect);
		    end;

        if PlayerShip.Living then
			    {move the players ship around}
			    MovePlayerShip
        else
        	{we died, start us over}
        	StartPlayer;

		    DrawPolygon(PlayerShip.ThePolygon,PlayerShip.NumVertices,PlayerShip.Angle,
		    						PlayerShip.XPos,PlayerShip.YPos,PlayerShip.Color);
		    if PlayerShip.ShieldsOn then
		    	with FOffscreenBuffer.Canvas do
			    begin
		    		Pen.Color:=clGreen;
		        Brush.Style:=bsClear;
		        Ellipse(Round(PlayerShip.XPos-PlayerShip.ColDelta-5),Round(PlayerShip.YPos-PlayerShip.ColDelta-5),
		        				Round(PlayerShip.XPos+PlayerShip.ColDelta+5),Round(PlayerShip.YPos+PlayerShip.ColDelta+5));
		  	  end;

		    {draw ship exhaust effect}
		    DrawShipExhaust;

		    {Move all active missles}
		    MoveMissles;

		    {Move all Asteroids}
		    MoveAsteroids;

	      {draw any explosions that have just occured}
		    DrawExplosions;

		  	{Copy the next frame to the screen}
		    PlayField.Canvas.Draw(0,0,FOffscreenBuffer);

	      {Display Score Changes}
	      Panel2.Caption:='Score: '+IntToStr(Score);

		    Application.ProcessMessages;
  		end
      else				{this does a slight pause in between levels}
      begin
      	{Start the overall level pause}
        LevelPauseCounter:=0;

        {kill any moving sprites}
        ClearAll;

        {increase the level}
        Inc(CurLevel);

        while LevelPauseCounter<LEVELPAUSE do
        begin
        	{increment the counter}
          Inc(LevelPauseCounter);

	    		{Erase the former frame, so we can begin a new one}
      		With FOffscreenBuffer.Canvas do
		  		begin
		  			Brush.Color:=clBlack;
      			Brush.Style:=bsSolid;
		  			Fillrect(Cliprect);
		  		end;

      		{Display a message on the screen}
      		with FOffscreenBuffer.Canvas do
      		begin
      			SetTextAlign(Handle, TA_CENTER);
      			Font.Name:='Arial';
      		  Font.Size:=30;
      		  Font.Color:=clRed;
      		  Brush.Style:=bsClear;

      		  {display the text centered in the screen}
      		  TextOut(FOffscreenBuffer.Width div 2,
      		  			 (FOffscreenBuffer.Height div 2)-(TextHeight('ABC')div 2),'LEVEL '+IntToStr(CurLevel));
      		end;

	  			{Copy the next frame to the screen}
	    		PlayField.Canvas.Draw(0,0,FOffscreenBuffer);
	    		Application.ProcessMessages;
        end;

        {start the player in the middle of the screen, with shields on}
				with PlayerShip do
  			begin
    			ShieldsOn:=True;
          ShieldLife:=0;
    			XPos:=316;
    			YPos:=204;
          XVel:=0;
          YVel:=0;
    			Angle:=0;
  			end;

  			{display the game values}
  			Panel2.Caption:='Score: '+IntToStr(Score);
  			if NumShipsLeft>-1 then 		{we don't want to display a negative amount of ships}
  				Panel3.Caption:='Lives Left: '+IntToStr(NumShipsLeft);
  			Panel4.Caption:='Level: '+IntToStr(CurLevel);

        {Now, generate some new asteroids, based on the level we are at}
        for Count:=0 to Random(5)+CurLevel do
        	StartAsteroid(Random(631),Random(408),Random(359),Random(10)+20);

        {this must be set so we enter the main playing loop}
        AnyAsteroidsMoved:=True;
      end;		{end the inter-level loop}
    end
    else   	{play some intermission stuff}
    while NumShipsLeft<0 do
    begin
     {Erase the former frame, so we can begin a new one}
	    With FOffscreenBuffer.Canvas do
	    begin
	    	Brush.Color:=clBlack;
	    	Brush.Style:=bsSolid;
	    	Fillrect(Cliprect);
	    end;

      {Move the random asteroids}
      MoveAsteroids;

      {Display a message on the screen}
      with FOffscreenBuffer.Canvas do
      begin
      	SetTextAlign(Handle, TA_CENTER);
      	Font.Name:='Arial';
        Font.Size:=30;
        Font.Color:=clRed;
        Brush.Style:=bsClear;

        {display the text centered in the screen}
        TextOut(FOffscreenBuffer.Width div 2,
        			 (FOffscreenBuffer.Height div 2)-(TextHeight('ABC')div 2),'ASTEROIDS');
      end;

	  	{Copy the next frame to the screen}
	    PlayField.Canvas.Draw(0,0,FOffscreenBuffer);
	    Application.ProcessMessages;
    end;
  end;
end;

procedure TAsteroidForm.DrawPolygon(const ThePolygon: Array of TRPoint; NumVerts: integer;
																		TheAngle: Integer; XPos, YPos: Real; TheColor: TColor);
var
	TempPoly: Array[0..19] of TPoint;
  Count: Integer;
begin
	{Rotate the polygon by the angle, and translate it's position}
  for Count:=0 to NumVerts-1 do
  begin
  	TempPoly[Count].X:=Round((ThePolygon[Count].X*CosineArray^[TheAngle]-ThePolygon[Count].Y*SineArray^[TheAngle])+XPos);
  	TempPoly[Count].Y:=Round((ThePolygon[Count].X*SineArray^[TheAngle]+ThePolygon[Count].Y*CosineArray^[TheAngle])+Ypos);
  end;

  {Now draw the actual polygon}
  FOffscreenBuffer.Canvas.Pen.Color:=TheColor;
  FOffscreenBuffer.Canvas.Brush.Style:=bsClear;
  FOffscreenBuffer.Canvas.Brush.Color:=clBlack;
  Polyline(FOffscreenBuffer.Canvas.Handle,TempPoly,NumVerts);
end;

function KeyboardHook(Code: Integer; WordParam: Word; LongParam: LongInt): LongInt;
begin
	case WordParam of
  	{start a bullet}
  	vk_Space: AsteroidForm.StartMissle(PlayerShip.XPos,PlayerShip.YPos,PlayerShip.Angle,True,
    																	 PlayerShip.XVel,PlayerShip.YVel);

    {rotation}
		vk_Right: PlayerShip.Angle:=PlayerShip.Angle+15;
    vk_Left:  PlayerShip.Angle:=PlayerShip.Angle-15;

    {accelerate}
    vk_Up:	begin
    					AsteroidForm.StartShipExhaustBurst;
   						PlayerShip.XVel:=PlayerShip.XVel+CosineArray^[PlayerShip.Angle];
              if PlayerShip.XVel>10 then PlayerShip.XVel:=10
              else
              if PlayerShip.XVel<-10 then PlayerShip.XVel:=-10;

   						PlayerShip.YVel:=PlayerShip.YVel+SineArray^[PlayerShip.Angle];
              if PlayerShip.YVel>10 then PlayerShip.YVel:=10
              else
              if PlayerShip.YVel<-10 then PlayerShip.YVel:=-10;
            end;

    {decelerate}
    vk_Down:	begin
   							PlayerShip.XVel:=PlayerShip.XVel-CosineArray^[PlayerShip.Angle];
              	if PlayerShip.XVel<0 then PlayerShip.XVel:=0;

   							PlayerShip.YVel:=PlayerShip.YVel-SineArray^[PlayerShip.Angle];
              	if PlayerShip.YVel<0 then PlayerShip.YVel:=0;
    					end;

    {turn the shields on}
    vk_Return: begin
    						PlayerShip.ShieldsOn:=True;
                PlayerShip.ShieldLife:=0;
    					 end;
  end;

  {adjust the angle if it has gone around one revolution}
  if PlayerShip.Angle>359 then PlayerShip.Angle:=PlayerShip.Angle-359;
  if PlayerShip.Angle<0 then PlayerShip.Angle:=360+PlayerShip.Angle;
end;

procedure TAsteroidForm.FormCreate(Sender: TObject);
var
	HookProc: TFarProc;
  Count, Count2: Integer;
begin
	FOffscreenBuffer:=TBitmap.Create;			{make the double buffer and initialize it}
  with FOffscreenBuffer do
  begin
  	Width:=632;
  	Height:=409;
  	FOffscreenBuffer.Canvas.Brush.Color:=clBlack;
    FOffscreenBuffer.Canvas.FillRect(Canvas.ClipRect);
  end;

  {initialize random number generation}
  Randomize;

  {Set the keyboard hook so we can intercept keyboard input. This is necessary as
   the TImage is not a windowed control and cannot recieve keyboard input, and
   KeyPreview for the form will not get cursor keys.}
  HookProc:=MakeProcInstance(@KeyboardHook, HInstance);
  KBHook:=SetWindowsHookEx(WH_KEYBOARD,THookProc(HookProc),HInstance,GetCurrentTask);

  {initialize the trigonometry lookup tables}
  InitSineArray;
  InitCosineArray;

  FDoLoop:=True;                        {Yes, let's start the main control loop}

  {initialize all global objects}
  PlayerShip:= TPlayerShip.Create;

  {so we start off with a new level}
  AnyAsteroidsMoved:=False;

  for Count:=0 to NUMASTEROIDS do
  	Asteroids[Count]:= TAsteroid.Create;

  for Count:=0 to NUMMISSLES do
		Missles[Count]:= TParticle.Create;

  for Count:=0 to NUMEXLPDS do
  	for Count2:=0 to NUMPARTICLES do
  		Explosions[Count].Particles[Count2]:= TParticle.Create;

  for Count:=0 to NUMPARTICLES do
    ShipExhaust[Count]:= TParticle.Create;

{Define what the players ship looks like}
with PlayerShip do
begin
	Color:=clYellow;

	NumVertices:=5;
  ColDelta:=5;					{this should be enough of a bounding box}
  ThePolygon[0].x:=6;
  ThePolygon[0].y:=0;

  ThePolygon[1].x:=-6;
  ThePolygon[1].y:=7;

  ThePolygon[2].x:=-4;
  ThePolygon[2].y:=0;

  ThePolygon[3].x:=-6;
  ThePolygon[3].y:=-7;

  ThePolygon[4].x:=6;
  ThePolygon[4].y:=0;
end;

{set the OnIdle event so the main loop will start automatically}
Application.OnIdle:=Idling;

{we start out in Intermission mode, so zero out everything}
EndGame;
end;

procedure TAsteroidForm.FormDestroy(Sender: TObject);
var
	Count, Count2: Integer;
begin
	FOffscreenBuffer.Free;				{clean up after ourselves}

  {Deallocate the trig lookup tables}
  DestroySineArray;
  DestroyCosineArray;

  {unhook the keyboard interception}
  UnHookWindowsHookEx(KBHook);

  {deinitialize all global objects}
  PlayerShip.Free;
  
  for Count:=0 to NUMASTEROIDS do
  	Asteroids[Count].Free;

  for Count:=0 to NUMMISSLES do
		Missles[Count].Free;

  for Count:=0 to NUMEXLPDS do
  	for Count2:=0 to NUMPARTICLES do
  		Explosions[Count].Particles[Count2].Free;

  for Count:=0 to NUMPARTICLES do
    ShipExhaust[Count].Free;
end;

procedure TAsteroidForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
{----- without this, our main loop would continue even if we   -----}
{----- tried to shut down the form, so we must tell it to stop -----}
	FDoLoop:=False;
  Halt;
end;

{Start a new asteroid in the asteroids array}
procedure TAsteroidForm.StartAsteroid(InitialX,InitialY: Integer; Facing: Integer;
																			Radius: Integer);
var
	Count1, Count2: Integer;
  AngleStep: Integer;
  CurAngle: Integer;
  Noise1, Noise2: integer;
begin
	for Count1:=0 to NUMASTEROIDS do
 		if not Asteroids[Count1].Living then
   	with Asteroids[Count1] do
    	begin
     		Living:=True;
      	Angle:=Facing;

        if Radius>MAXASTEROIDRAD then Radius:=MAXASTEROIDRAD;
        if Radius<5 then Radius:=5;
      	ColDelta:=Radius;	{relative bounding rectangle}

        {Determine a random velocity}
      	XVel:=CosineArray^[Angle]*(random(3)+(MAXASTEROIDRAD div Radius));
      	YVel:=SineArray^[Angle]*(random(3)+(MAXASTEROIDRAD div Radius));

        {Determine a random rotation rate}
        RotationRate:=random(20)-10;

      	{place the asteroid}
      	XPos:=InitialX;
      	YPos:=InitialY;

      	{Now, let's create the asteroid polygon}
        NumVertices:=Random(14)+5;

        {this is how many degress we can rotate through for each vertex}
        AngleStep:=359 div NumVertices;
        CurAngle:=0;

        {create some vertices in a rough circle with varying magnitudes from the origin}
        for Count2:=0 to NumVertices-1 do
        begin
        	CurAngle:=CurAngle+AngleStep;
          Noise1:=Random(10)-5;
          Noise2:=Random(10)-5;
          ThePolygon[Count2].X:=CosineArray^[CurAngle]*Radius+Noise1;
          ThePolygon[Count2].Y:=SineArray^[CurAngle]*Radius+Noise2;
        end;

        {close the polygon}
        ThePolygon[Count2].X:=ThePolygon[0].X;
        ThePolygon[Count2].Y:=ThePolygon[0].Y;

    		break;
			end;
end;

procedure TAsteroidForm.MoveAsteroids;
var
	Count, Count2, Count3: Integer;
  Direction: Integer;
  IntersectRect: TRect;				{this is the coordinates of the intersection rectangle}
begin
	{start checking for any living asteroids}
  AnyAsteroidsMoved:=False;

	for Count:=0 to NUMASTEROIDS do
  begin
  	if Asteroids[Count].Living then
    begin
    	{there is at least one asteroid still left alive}
      AnyAsteroidsMoved:=True;

    	{add the velocities to the position}
			Asteroids[Count].XPos:=Asteroids[Count].XPos+Asteroids[Count].XVel;
			Asteroids[Count].YPos:=Asteroids[Count].YPos+Asteroids[Count].YVel;

      {check for screen sides}
    	if Asteroids[Count].XPos>632 then Asteroids[Count].XPos:=0;
    	if Asteroids[Count].XPos<0 then Asteroids[Count].XPos:=632;
    	if Asteroids[Count].YPos>409 then Asteroids[Count].YPos:=0;
    	if Asteroids[Count].YPos<0 then Asteroids[Count].YPos:=409;

      {This is the general collision detection code.  We must see if we have collided
       with either a missle or the players ship.  We will start by checking our
       bounding rectangle against the players bounding rectangle.  If there is a collision,
       we kill the player, kill this asteroid, start two explosions, and create some
       more asteroids, only a little smaller}
      if (WinProcs.IntersectRect(IntersectRect,
      										 			Rect(Round(Asteroids[Count].XPos-Asteroids[Count].ColDelta),
                           					 Round(Asteroids[Count].YPos-Asteroids[Count].ColDelta),
                           					 Round(Asteroids[Count].XPos+Asteroids[Count].ColDelta),
                                		 Round(Asteroids[Count].YPos+Asteroids[Count].ColDelta)),
      										 			Rect(Round(PlayerShip.XPos-PlayerShip.ColDelta),
                                		 Round(PlayerShip.YPos-PlayerShip.ColDelta),
                                		 Round(PlayerShip.XPos+PlayerShip.ColDelta),
                                     Round(PlayerShip.YPos+PlayerShip.ColDelta)))>0)
      		and not PlayerShip.ShieldsOn then
      begin				{if there has been a collision...}
         StartExplosion(Round(PlayerShip.XPos), Round(PlayerShip.YPos));
         StartExplosion(Round(Asteroids[Count].XPos), Round(Asteroids[Count].YPos));
         Asteroids[Count].Living:=False;
         PlayerShip.Living:=False;				{kill this player}

         {increase the players score. this should give us a higher score for smaller asteroids}
         Score:=Score+(MAXASTEROIDRAD-Asteroids[Count].ColDelta);

         for Count3:=0 to Random(3) do						{generate a few asteroids...}
         		if Asteroids[Count].ColDelta>10 then	{but only if it's not too small}
         			StartAsteroid(Round(Asteroids[Count].XPos),Round(Asteroids[Count].YPos),
         							 			Random(359),Asteroids[Count].ColDelta-5);
      end;

      {now, test against all of the bullets}
			for Count2:=0 to NUMMISSLES do
      	if Missles[Count2].Living=True then
      	if WinProcs.IntersectRect(IntersectRect,
      											 			Rect(Round(Asteroids[Count].XPos-Asteroids[Count].ColDelta),
          	                 					 Round(Asteroids[Count].YPos-Asteroids[Count].ColDelta),
            	               					 Round(Asteroids[Count].XPos+Asteroids[Count].ColDelta),
              	                  		 Round(Asteroids[Count].YPos+Asteroids[Count].ColDelta)),
      											 			Rect(Round(Missles[Count2].XPos-Missles[Count2].ColDelta),
                  	              		 Round(Missles[Count2].YPos-Missles[Count2].ColDelta),
                    	            		 Round(Missles[Count2].XPos+Missles[Count2].ColDelta),
                      	               Round(Missles[Count2].YPos+Missles[Count2].ColDelta)))>0 then
      		begin						{we shot an asteroid}
		         StartExplosion(Round(Missles[Count2].XPos), Round(Missles[Count2].YPos));
    		     Asteroids[Count].Living:=False;
             Missles[Count2].Living:=False;


  		       {increase the players score. this should give us a higher score for smaller asteroids}
	    	     Score:=Score+(MAXASTEROIDRAD-Asteroids[Count].ColDelta);

		         for Count3:=0 to Random(4) do						{generate a few asteroids...}
		         		if Asteroids[Count].ColDelta>10 then	{but only if it's not too small}
		         			StartAsteroid(Round(Asteroids[Count].XPos),Round(Asteroids[Count].YPos),
		         							 			Random(359),Asteroids[Count].ColDelta-5);
          end;

      {Modify its rotation}
      Asteroids[Count].Angle:=Asteroids[Count].Angle+Asteroids[Count].RotationRate;

  		{adjust the angle if it has gone around one revolution}
  		if Asteroids[Count].Angle>359 then Asteroids[Count].Angle:=Asteroids[Count].Angle-359;
  		if Asteroids[Count].Angle<0 then Asteroids[Count].Angle:=360+Asteroids[Count].Angle;

      {Now, draw the polygon}
    	DrawPolygon(Asteroids[Count].ThePolygon,Asteroids[Count].NumVertices,Asteroids[Count].Angle,
      						Asteroids[Count].XPos,Asteroids[Count].YPos,clBlue);
    end;
  end;
end;

{start a new player, in the middle of the game field, with shields on}
procedure TAsteroidForm.StartPlayer;
begin
	with PlayerShip do
  begin
  	Living:=True;
    ShieldsOn:=True;
    ShieldLife:=0;
    XPos:=316;
    YPos:=204;
    XVel:=0;
    YVel:=0;
    Angle:=0;
  end;

  {subtract one from the overall ships left}
  Dec(NumShipsLeft);

  {display the game values}
  Panel2.Caption:='Score: '+IntToStr(Score);
  if NumShipsLeft>-1 then 		{we don't want to display a negative amount of ships}
  	Panel3.Caption:='Lives Left: '+IntToStr(NumShipsLeft);
  Panel4.Caption:='Level: '+IntToStr(CurLevel);
end;

{move the players ship}
procedure TAsteroidForm.MovePlayerShip;
begin
	{Add velocities to current position}
  with PlayerShip do
  begin
  	XPos:=XPos+XVel;
    YPos:=YPos+YVel;

    {if the shields are on, increment their lifespan count}
    if ShieldsOn then
	    ShieldLife:=ShieldLife+1;

    {if the shields have worn out, turn them off}
    if ShieldLife>SHIELDLIFESPAN then
    begin
    	ShieldsOn:=False;
    end;

    {check for edge of screen}
    if XPos>632 then XPos:=0;
    if XPos<0 then XPos:=632;
    if YPos>409 then YPos:=0;
    if YPos<0 then YPos:=409;
  end;
end;

{This starts some particles in the ship exhaust particle system}
procedure TAsteroidForm.StartShipExhaustBurst;
var
	Count1, Count2: Integer;
  Angle: Integer;
begin
	{Start a random number of particles}
  for Count1:=0 to random(4)+5 do
  	for Count2:=0 to NUMPARTICLES do
    	if not ShipExhaust[Count2].Living then
				with ShipExhaust[Count2] do
        begin
        	Living:=true;

          {opposite facing of the player plus a small random amount}
          Angle:=PlayerShip.Angle+180+(random(30)-15);

          {make sure our angle is valid}
  				if Angle>359 then Angle:=Angle-359;
  				if Angle<0 then Angle:=360+Angle;

          {determine lifespan, plus a random amount}
          LifeSpan:=EXHAUSTLIFESPAN+(random(10)+1);

        	{This will insure that the exhaust is always slower than the player's ship}
        	XVel:=PlayerShip.XVel+(CosineArray^[Angle]*EXHAUSTVELDELTA);
        	YVel:=PlayerShip.YVel+(SineArray^[Angle]*EXHAUSTVELDELTA);

        	{Start the exhaust a little bit behind the ship}
        	XPos:=PlayerShip.XPos+(CosineArray^[Angle]*4);
        	YPos:=PlayerShip.YPos+(SineArray^[Angle]*4);

          break;
        end;
end;

{this procedure moves all of the live missles, draws them to the offscreen buffer,
 performs collision checks, etc.}
procedure TAsteroidForm.DrawShipExhaust;
var
	Count: Integer;
  ParticleColor: TColor;
  ColorIndex: Real;
begin
	for Count:=0 to NUMPARTICLES do
  begin
  	{add the velocities to the position}
  	ShipExhaust[Count].XPos:=(ShipExhaust[Count].XPos+ShipExhaust[Count].XVel);
  	ShipExhaust[Count].YPos:=(ShipExhaust[Count].YPos+ShipExhaust[Count].YVel);

    {Decrease the LifeSpan, as another animation frame has gone by}
    ShipExhaust[Count].LifeSpan:=ShipExhaust[Count].LifeSpan-1;

    {clip to the sides of the screen and range}
    if (ShipExhaust[Count].LifeSpan=0) or (ShipExhaust[Count].XPos>632) or (ShipExhaust[Count].XPos<0) or
			 (ShipExhaust[Count].YPos>409) or (ShipExhaust[Count].YPos<0)
    	then ShipExhaust[Count].Living:=False;

    {if the exhaust particle is still alive, then draw it}
    if ShipExhaust[Count].Living then
    begin
			{determine where the color will fall in the color array}
      ColorIndex:=EXHAUSTLIFESPAN/5;	{we have 5 colors}

      {and the particle color is...}
      ParticleColor:=FadeColors[Round(ShipExhaust[Count].LifeSpan/ColorIndex)];

    	FOffscreenBuffer.Canvas.Pixels[Round(ShipExhaust[Count].XPos),Round(ShipExhaust[Count].YPos)]:=ParticleColor;
    end;
  end;
end;

{This procedure starts bullets for player and enemy alike, if one is available. Only
 the enemy can use the last two bullet slots in the array}
procedure TAsteroidForm.StartMissle(InitialX, InitialY: Real; Facing: Integer; IsPlayer: Boolean;
																		StartXVel, StartYVel: Real);
var
  Count: Integer;					{general loop counter}
  MaxSlots: Integer;			{the maximum slots available}
begin
  if IsPlayer then MaxSlots:=NUMMISSLES-2 else MaxSlots:=NUMMISSLES;	{reserve the last two for enemy only}

  {loop through the Missles array. If we find a slot that does not have a missle in it,
   we can fill it with our missle}
  for Count:=0 to MaxSlots do
		if not Missles[Count].Living then
    	with Missles[Count] do
      begin

      	Living:=True;
        ColDelta:=2;	{small bounding rectangle}

        {LifeSpan measures how long they will live, in terms of number
         of frames that will pass before they fade away. This will give them a specific
         range instead of the entire board.}
        LifeSpan:= MISSLEFRAMELIFE;

        {This will insure that the bullet is always faster than the player's ship}
        XVel:=StartXVel+(CosineArray^[Facing]*MISSLEVELDELTA);
        YVel:=StartYVel+(SineArray^[Facing]*MISSLEVELDELTA);

        {Start the bullet a little bit ahead of the ship}
        XPos:=InitialX+CosineArray^[Facing]*2;
        YPos:=InitialY+SineArray^[Facing]*2;

    		break;
      end;
end;

{this procedure moves all of the live missles, draws them to the offscreen buffer,
 performs collision checks, etc.}
procedure TAsteroidForm.MoveMissles;
var
	Count: Integer;
  MissleColor: TColor;
  ColorIndex: Real;
begin
	for Count:=0 to NUMMISSLES do
  begin
  	{add the velocities to the position}
  	Missles[Count].XPos:=(Missles[Count].XPos+Missles[Count].XVel);
  	Missles[Count].YPos:=(Missles[Count].YPos+Missles[Count].YVel);

    {Decrease the LifeSpan, as another animation frame has gone by}
    Missles[Count].LifeSpan:=Missles[Count].LifeSpan-1;

    {clip to the range}
    if (Missles[Count].LifeSpan=0) then
    	Missles[Count].Living:=False;

    {check for screen sides}
  	if Missles[Count].XPos>632 then Missles[Count].XPos:=0;
   	if Missles[Count].XPos<0 then Missles[Count].XPos:=632;
   	if Missles[Count].YPos>409 then Missles[Count].YPos:=0;
   	if Missles[Count].YPos<0 then Missles[Count].YPos:=409;

    {if the missle is still alive, then draw it}
    if Missles[Count].Living then
    begin
			{determine where the color will fall in the color array}
      ColorIndex:=MISSLEFRAMELIFE/5;	{we have 5 colors}

      {and the particle color is...}
      MissleColor:=FadeColors[Round(Missles[Count].LifeSpan/ColorIndex)];

			FOffscreenBuffer.Canvas.Pen.Color:=MissleColor;
    	FOffscreenBuffer.Canvas.Brush.Style:=bsClear;
    	FOffscreenBuffer.Canvas.Ellipse(Round(Missles[Count].XPos-1),Round(Missles[Count].YPos-1),
    		 														  Round(Missles[Count].XPos+1),Round(Missles[Count].YPos+1));
    end;
  end;
end;

{this procedure parses through the explosion list. if a free explosion slot is found,
 it generates all the particles for that explosion}
procedure TAsteroidForm.StartExplosion(InitialX,InitialY: Integer);
var
	Count1, Count2: Integer;
  CurAngle, AngleStep: Integer;
begin
	{see if an explosion slot is available}
	for Count1:=0 to NUMEXLPDS do
  	if not Explosions[Count1].Living then
    begin
      Explosions[Count1].Living:=True;  {Flag this spot as taken}

      {prepare to generate the particles radiating from the center outwards
       in a 360 degree circle}
      CurAngle:=0;
      AngleStep:=360 div NUMPARTICLES;

 			{now, step through each particle, setting it up accordingly}
      for Count2:=0 to NUMPARTICLES do
      begin
      	{flag the particle as alive}
				Explosions[Count1].Particles[Count2].Living:=True;

      	{determine lifespan, plus a random amount}
        Explosions[Count1].Particles[Count2].LifeSpan:=EXPLDLIFE+(random(10)+1);

        {determine the velocity, plus a little extra}
        Explosions[Count1].Particles[Count2].XVel:=CosineArray^[CurAngle]*EXPLDVELDELTA*(Random(2)+1);
				Explosions[Count1].Particles[Count2].YVel:=SineArray^[CurAngle]*EXPLDVELDELTA*(Random(2)+1);

        {Start the exhaust a little bit behind the ship}
        Explosions[Count1].Particles[Count2].XPos:=InitialX;
        Explosions[Count1].Particles[Count2].YPos:=InitialY;

        {increase our position around the circle}
        CurAngle:=CurAngle+AngleStep;
      end;

    	Break;
    end;
end;

procedure TAsteroidForm.DrawExplosions;
var
	Count1, Count2, DeadParticles: Integer;
  ParticleColor: TColor;
  ColorIndex: Real;
begin
	{check to see if we need to play this explosion}
	for Count1:=0 to NUMEXLPDS do
  	if Explosions[Count1].Living then
    begin
    {this tallies all of the dead particles in the explosion. if they are all
     dead, then this explosion slot can be freed.}
     DeadParticles:=0;

     {parse all the particles and draw them}
     for Count2:=0 to NUMPARTICLES do
     begin
     		with Explosions[Count1].Particles[Count2] do
        begin
			  	{add the velocities to the position}
			  	XPos:=XPos+XVel;
			  	YPos:=YPos+YVel;

			    {Decrease the LifeSpan, as another animation frame has gone by}
			    LifeSpan:=LifeSpan-1;

			    {clip to the sides of the screen and range}
			    if (LifeSpan=0) or (XPos>632) or (XPos<0) or
						 (YPos>409) or (YPos<0)
			    	then
            begin
            	Living:=False;
              Inc(DeadParticles);
            end;

			    {if the explosion particle is still alive, then draw it}
			    if Living then
			    begin
						{determine where the color will fall in the color array}
			      ColorIndex:=EXPLDLIFE/5;	{we have 5 colors}

			      {and the particle color is...}
			      ParticleColor:=FadeColors[Round(LifeSpan/ColorIndex)];

			    	FOffscreenBuffer.Canvas.Pixels[Round(XPos),Round(YPos)]:=ParticleColor;
	    		end;
	  		end;
     end;

    {check to see if all the particles in this explosion are dead, and if so, free the slot}
    if DeadParticles>=NUMPARTICLES-10 then
    	Explosions[Count1].Living:=False;

    end;
end;

procedure TAsteroidForm.NewGame1Click(Sender: TObject);
begin
	{Make sure any random asteroids are all dead}
  ClearAll;

  {initialize the global variables for score, lives, and level}
  NumShipsLeft:=3;
  Score:=0;
  CurLevel:=0;

  {Start a new player ship}
  StartPlayer;

  {go back into the main loop}
	MainLoop;
end;

{Show the about screen}
procedure TAsteroidForm.About1Click(Sender: TObject);
begin
{pause the main loop while the about box is up}
FDoLoop:=False;

{show the about box}
AboutForm:=TAboutForm.Create(Application);
AboutForm.ShowModal;
AboutForm.Free;

{Restart main loop}
FDoLoop:=True;
MainLoop;
end;

{kick us into the main loop when the app becomes idle}
procedure TAsteroidForm.Idling(Sender: TObject; var Done: Boolean);
begin
	MainLoop;
  Done:=true;
end;

{this will zero out all of the arrays, which will be usefull when we want to start another
 game or another level}
procedure TAsteroidForm.ClearAll;
var
	Count: Integer;
begin
	{clear all asteroids}
	for Count:=0 to NUMASTEROIDS do
  	Asteroids[Count].Living:=False;

  {clear all missles}
  for Count:=0 to NUMMISSLES do
  	Missles[Count].Living:=False;

  {clear all explosions}
  for Count:=0 to NUMEXLPDS do
  	Explosions[Count].Living:=False;
end;

{this is a general procedure that will end the current game, generate some random asteroids,
 and put us into Intermission mode}
procedure TAsteroidForm.EndGame;
var
	Count: Integer;
begin
NumShipsLeft:=-1;
Score:=0;
CurLevel:=0;

{display values}
Panel2.Caption:='Score: '+IntToStr(Score);
Panel3.Caption:='Lives Left: 0';
Panel4.Caption:='Level: '+IntToStr(CurLevel);

{make sure everything is dead}
ClearAll;

{Generate some random asteroids}
for Count:=0 to 10 do
	StartAsteroid(Random(FOffscreenBuffer.Width),Random(FOffscreenBuffer.Height),
  							Random(359),Random(MAXASTEROIDRAD));

end;

{stop the program.}
procedure TAsteroidForm.Exit1Click(Sender: TObject);
begin
Application.OnIdle:=nil;
FDoLoop:=False;
Halt;
end;

end.
